// EMERGENT GAME TECHNOLOGIES PROPRIETARY INFORMATION
//
// This software is supplied under the terms of a license agreement or
// nondisclosure agreement with Emergent Game Technologies and may not 
// be copied or disclosed except in accordance with the terms of that 
// agreement.
//
//      Copyright (c) 1996-2007 Emergent Game Technologies.
//      All Rights Reserved.
//
// Emergent Game Technologies, Chapel Hill, North Carolina 27517
// http://www.emergent.net

#ifndef NISHAREDDATALIST_H
#define NISHAREDDATALIST_H

#include "NiPluginToolkitLibType.h"
#include <NiTMap.h>
#include <NiCriticalSection.h>
#include "NiSharedData.h"

/// This class manages the currently available shared data.
class NIPLUGINTOOLKIT_ENTRY NiSharedDataList : public NiMemObject
{
public:
    /// A return code for list member functions.
    enum ErrorCode
    {
        SUCCESS = 0,
        DUPLICATE_OBJECT,
        OBJECT_NOT_FOUND,
        CURRENT_THREAD_HAS_NOT_LOCKED
    };

    /// The public singleton creator.
    static void CreateInstance();

    /// The public singleton destroyer.
    static void DestroyInstance();
    
    /// The public singleton accessor.
    /// @return A pointer to the one and only instance of the class or
    ///         NULL if no instance exists.
    static NiSharedDataList* GetInstance();

    /// Static helper function that locks the shared data list. Asserts if
    /// the shared data list does not exist.
    static void LockSharedData();

    /// Static helper function that unlocks the shared data list. Asserts if
    /// the shared data list does not exist.
    static void UnlockSharedData();
    
    /// Locks shared data to provide thread-safe access. All threads spawned
    /// from plug-ins must call this before accessing shared data.
    void Lock();

    /// Unlocks shared data to provide thread-safe access. All threads spawned
    /// from plug-ins must call this when done accessing shared data.
    void Unlock();

    /// Inserts a shared data object into the list.
    ErrorCode Insert(NiSharedData* pkNewData);

    /// Removes the shared data object specified by the given type from the
    /// list.
    ErrorCode Delete(NiSharedDataType pkType);

    /// Removes all the shared data objects from the list.
    ErrorCode DeleteAll();

    /// Gets the shared data object specified by the RTTI object from the
    /// list. If the returned value is NULL, call GetLastError to find out 
    /// what went wrong.
    NiSharedData* Get(NiSharedDataType pkType);

    /// The last error code generated by this object.
    ErrorCode GetLastError();

    /// Get the total number of shared data pieces
    unsigned int GetSharedDataCount();

    /// Get the piece of shared data at a particular location
    NiSharedData* GetSharedDataAt(unsigned int uiWhichData);

protected:
    /// Protected default constructor. This is protected to enforce the
    /// Singleton pattern.
    NiSharedDataList(){}

    /// Protected copy constructor. This is protected to enforce the
    /// Singleton pattern.
    NiSharedDataList(NiSharedDataList&){}

    /// Protected destructor. This is protected to enforce the Singleton
    /// pattern.
    ~NiSharedDataList(){}

    /// A critical section for use with locking and unlocking the list.
    NiCriticalSection m_kCriticalSection;

    /// The collection of shared data objects, indexed by type.
    NiTMap<NiSharedDataType, NiSharedDataPtr> m_kSharedDataObjects;

    /// The singleton instance.
    static NiSharedDataList* ms_pkThis;

    /// The last error code
    ErrorCode m_eLastError;
};

#endif  // #ifndef NISHAREDDATALIST_H
