﻿// Copyright 2018 RED Software, LLC. All Rights Reserved.

using System.Collections.Generic;

namespace IgniteEngine.Networking
{
    /// <summary>
    /// The callback type for a network message handler.
    /// </summary>
    /// <param name="message">The message being handled.</param>
    /// <param name="connection">The connection that sent the message.</param>
    public delegate void NetworkMessageHandlerMethod(NetworkMessage message, NetworkConnection connection);

    /// <summary>
    /// Class to register handlers for network messages.
    /// </summary>
    public class NetworkMessageHandler
    {
        /// <summary>
        /// All registered message handlers.
        /// </summary>
        private static readonly Dictionary<NetworkCommand, NetworkMessageHandlerMethod> handlers = new Dictionary<NetworkCommand, NetworkMessageHandlerMethod>();

        /// <summary>
        /// Registers a message handler to the collection of handlers.
        /// </summary>
        /// <param name="command">The command being handled.</param>
        /// <param name="method">The method to invoke.</param>
        public static void Register(NetworkCommand command, NetworkMessageHandlerMethod method)
        {
            if (!handlers.ContainsKey(command))
            {
                handlers.Add(command, method);
            }
        }

        /// <summary>
        /// Tries to get a message handler from the collection.
        /// </summary>
        /// <param name="command">The message's command.</param>
        /// <param name="method">The handler method.</param>
        /// <returns></returns>
        public static bool TryFetch(NetworkCommand command, out NetworkMessageHandlerMethod method)
        {
            return handlers.TryGetValue(command, out method);
        }
    }
}
